C======================================================================
C This code estimates parameters of the Brooks-Corey and van Genuchten 
C water retention models from sand, silt, clay, OC contents, soil bulk and particle 
C densities using 16 pedotransfer functions. 
C The code returns "-1" for those parameters which can not be computed 
C due to lack of input data. 
C
C  The last four PTF estimate water content at capillary pressure of 330 and 15000 cm.
C The unput data are: 
C ID, Depth (cm), Sand (%), Silt (%), Clay (%),  OC (%), BD (g/cm3), PD (g/cm3)
C
C=============================================================
C||     Guber, A.K., Pachepsky Y.A. USDA-ARS-EMFSL, Beltsville, MD     ||
C||   Last corrections and modifications - November 2010.	            ||
C=============================================================
C
      Real P(9),Theta(14),Prawls02(11),Prawls03(10),Pvereecken(9),
     1 Prajkai(8),Pgupta(13),Ptomasella(9),Pmayr(7),OClevels(2),
     2 Depth(100),Sand(100),Silt(100),Clay(100),OC(100),BD(100),
     3 Pwosten(14), PD(100), SavePar(200), PB(6),PG(6),HyPar(16,100,4)
	Integer IM(21,100),ID(100)
	Equivalence (PB(1),ThetaR),(PB(2),Phi),(PB(3),hb),
     &            (PB(4),BCLambda),(PB(5),Gcap)
	Equivalence (PG(1),vgthr),(PG(2),vgths),(PG(3),vgalpha),
     &            (PG(4),vgn),(PG(5),vgm),(PG(6),Gcap)
	Character*30 Title(21)
	Data Title/"Saxton et al., 1986","Campbell and Shiosawa, 1992",
     1          "Rawls and Brakensiek, 1985","Williams et al., 1992",  
	2          "Williams et al., 1992","Oosterveld and Chang, 1980",
     3          "Mayr and Jarvice, 1999","Wosten et al., 1999",
     4          "Varallyay et al., 1982","Vereecken et al., 1989",
     5          "Wosten et al., 1999","Tomasella and Hodnett, 1998",
     6          "Rawls et al., 1982","Gupta and Larson, 1979",
     7          "Rajkai and Varallyay, 1992","Rawls et al., 1983",
     8          "Peterson et al., 1968","Bruand et al., 1994",
     9          "Canarache, 1993","Hall et al., 1977","Baumer, 1992"/

	Data Prawls02/0,100,200,330,600,1000,2000,4000,7000,10000,15000/
	Data P/10.,33.,50.,100.,330.,500.,1000.,3300.,15000./
	Data Prawls03/0,200,300,600,1000,2000,4000,7000,10000,15000/
	Data Pvereecken/0,3.2,10,32,100,200,630,2512,15864/
	Data Prajkai/0,3,10,32,500,2512,15849,1259000/
	Data Pgupta/0,40,70,100,200,330,600,1000,2000,4000,7000,10000,
     &            15000/
	Data Ptomasella/0,10,30,60,100,330,1000,5000,15000/
	Data Pmayr/2.5,20,51,102,408,2040,15300/
	Data Pwosten/0,10,20,50,100,200,250,500,1000,2000,5000,10000,
     A             15000,16000/

	
      Open(10,file='C:\Ptf\ptf.in')
	Open(11,file='C:\Ptf\WR.par')
	Open(12,file='C:\Ptf\WC.out')
      Open(13,file='C:\Ptf\temp.dat')
	Do i=1,100
	Read(10,*,END=40,ERR=40) ID(i),Depth(i),Sand(i),Silt(i),
     *    Clay(i),OC(i),
     *	BD(i),PD(i)
		If(PD(i)<0) PD(i)=2.65
	end do
40	nd=i-1
	do j=1,nd
	  do i=1,21
	     IM(i,j)=1
        end do
      end do
C========================
C  PTF estimation begins
C=========================

C PTF1 Saxton et al, 1986
	Do j=1,nd
	  IS=1
	  If(Clay(j)<0.or.Sand(j)<0.or.BD(j)<0) then
	  call nodata(ThetaR,Phi,hb,BCLambda,a,Gcap,IM,1,j)
								  else
	  Call Saxton(Clay(j),Sand(j),A,B)
 	  Phi=1-BD(j)/PD(j)
	  ThetaR=0
	  BCLambda=-1./B
	  hb=10.*A*Phi**B
 	  Gcap=BCG(hb,BCLambda)
	  end if
	  call svpar(PB,IS,SavePar)

C PTF2 Campbell Campbell and Shiozawa, 1992
	  If(Clay(j)<0.or.Sand(j)<0.or.BD(j)<0) then
	  call nodata(ThetaR,Phi,hb,BCLambda,a,Gcap,IM,2,j)
											else
	  Call Campbell(Clay(j),Sand(j),BD(j),PD(j),ps,Phi,b)
	  ThetaR=0
	  BCLambda=1./b
	  hb=100*ps
	  Gcap=BCG(hb,BCLambda)
	  end if
	  call svpar(PB,IS,SavePar)

C PTF3 Rawls and Brakensiek, 1985	
	  If(Clay(j)<0.or.Sand(j)<0.or.BD(j)<0) then
	  call nodata(ThetaR,Phi,hb,BCLambda,a,Gcap,IM,3,j)
									else
	  Call Rawls85(Clay(j),Sand(j),BD(j),PD(j),Phi,BCLambda,hb,ThetaR)
	  Gcap=BCG(hb,BCLambda)
	  end if
	  call svpar(PB,IS,SavePar)
	
C PTF4 Williams 1992(1), Brooks&Corey model
	  If(Clay(j)<0.or.Sand(j)<0.or.BD(j)<0) then
	  call nodata(ThetaR,Phi,hb,BCLambda,a,Gcap,IM,4,j)
									else
	  Call Williams1(Clay(j),Sand(j),BD(j),2.0,A,B)
        ThetaR=0
	  Phi=1-BD(j)/PD(j)
	  BCLambda=-B
	  hb=1000*exp((alog(100*Phi)-A)/B)
	  Gcap=BCG(hb,BCLambda)
	  end if
	  call svpar(PB,IS,SavePar)

C PTF5 Williams 1992(2), Brooks&Corey model
	  If(Clay(j)<0.or.Sand(j)<0.or.OC(j)<0.or.BD(j)<0) then
	  call nodata(ThetaR,Phi,hb,BCLambda,a,Gcap,IM,5,j)
									else
	  Call Williams2(Clay(j),Sand(j),OC(j),A,B)
        ThetaR=0
	  Phi=1-BD(j)/PD(j)
	  BCLambda=-B
	  hb=1000*exp((alog(100*Phi)-A)/B)
	  Gcap=BCG(hb,BCLambda)
	  end if
	  call svpar(PB,IS,SavePar)

C PTF6 Oosterveld and Chang, 1980. Brooks&Corey WR model
	  If(Clay(j)<0.or.Sand(j)<0.or.BD(j)<0.or.Depth(j)<0) then
	  call nodata(ThetaR,Phi,hb,BCLambda,a,Gcap,IM,6,j)
									else
	  Call Oosterveld(Clay(j),Sand(j),BD(j),Depth(j),A)
	  ThetaR=0
	  Phi=1-BD(j)/PD(j)
	  BCLambda=0.19
	  hb=10*(A/Phi)**(1/0.19)
	  Gcap=BCG(hb,BCLambda)
	  end if
	  call svpar(PB,IS,SavePar)

C PTF7 Mayr-Jarvis, 1999
      
	 If(Clay(j)<0.or.Silt(j)<0.or.Sand(j)<0.or.BD(j)<0.or.OC(j)<0)
     &													then
	  call nodata(ThetaR,Phi,hb,BCLambda,a,Gcap,IM,7,j)
														else
	  Call Mayr(Clay(j),Silt(j),Sand(j),BD(j),OC(j),hb,b,Phi)
	  BCLambda=1/b
	  Gcap=BCG(hb,BCLambda)
	  end if
	  call svpar(PB,IS,SavePar)

C PTF8 Wosten et al., 1999(1), van Genuchten model
	  If(Clay(j)<0.or.Sand(j)<0.or.Silt(j)<0) then
	  call nodata(vgthr,vgths,vgalpha,vgn,vgm,Gcap,IM,8,j)
												else
	  itop = 1
	  If(Depth(j)>30) itop = 0
	  Call Wosten1(Clay(j),Silt(j),Sand(j),itop,vgths,vgthr,vgalpha,
     1				vgn,vgm)
	  Gcap=vGG(vgalpha,vgm)
	  end if
	  call svpar(PG,IS,SavePar)


C PTF9 Varallyay et al, 1982
	  If(Clay(j)<0.or.Sand(j)<0.or.BD(j)<0) then
	  call nodata(vgthr,vgths,vgalpha,vgn,vgm,Gcap,IM,9,j)
											else
	  Call Varallyay(Clay(j),Sand(j),BD(j),vgths,vgn,pfx)
        vgalpha=10.**(-pfx)
	  vgm=1.0
	  vgthr=0.0
	  Gcap=-1.0
	  end if
	  call svpar(PG,IS,SavePar)

C PTF10 Vereecken et al. 1989
	  If(Clay(j)<0.or.Sand(j)<0.or.BD(j)<0.or.OC(j)<0) then
	  call nodata(vgthr,vgths,vgalpha,vgn,vgm,Gcap,IM,10,j)
	   												  else
	  call Vereckeen(Clay(j),Sand(j),BD(j),OC(j)
     &  ,vgths,vgthr,vgalpha,vgn)
	  end if
	  call svpar(PG,IS,SavePar)


C PTF11 Wosten et al., 1999(2), van Genuchten model
	  If(Clay(j)<0.or.Silt(j)<0.or.BD(j)<0.or.OC(j)<0) then
	  call nodata(vgthr,vgths,vgalpha,vgn,vgm,Gcap,IM,11,j)
														else
	  itop = 1
	  If(Depth(j)<30) itop = 0
	  Call 	Wosten2(Clay(j),Silt(j),BD(j),OC(j),
     *		  itop,vgths,vgthr,vgalpha,vgn)
	  vgm=1-1/vgn
	  Gcap=vGG(vgalpha,vgm)
	  end if
	  call svpar(PG,IS,SavePar)


C PTF12 Tomasella & Hodnett, 1998
	  If(Clay(j)<0.or.Silt(j)<0.or.OC(j)<0) then
	  call nodata(vgthr,vgths,vgalpha,vgn,vgm,Gcap,IM,12,j)
											  else
	  Call Tomasella(Clay(j),Silt(j),OC(j),Theta)
	  call VGpar(Ptomasella,Theta,9,PG,4,50)
	  vgm=1-1/vgn
	  Gcap=vGG(vgalpha,vgm)
	  end if
	  call svpar(PG,IS,SavePar)

C PTF13 Rawls, Brakensiek, Saxton 1982
	If(Clay(j)<0.or.Silt(j)<0.or.Clay(j)<0.or.OC(j)<0.or.BD(j)<0) then
	  call nodata(vgthr,vgths,vgalpha,vgn,vgm,Gcap,IM,13,j)
											  else
	  Call Rawls82(Clay(j),Silt(j),Sand(j),OC(j),Theta)
	  Theta(1)=1-BD(j)/PD(j)
	  call VGpar(PRawls02,Theta,11,PG,4,50)
	  vgm=1-1/vgn
	  Gcap=vGG(vgalpha,vgm)
	  end if
	  call svpar(PG,IS,SavePar)

C PTF14 Gupta and Larson, 1979
	  If(Clay(j)<0.or.Silt(j)<0.or.Sand(j)<0.or.OC(j)<0.
     &	  or.BD(j)<0) then
	  call nodata(vgthr,vgths,vgalpha,vgn,vgm,Gcap,IM,14,j)
											  else
	  Call GuLar(Clay(j),Silt(j),Sand(j),OC(j),BD(j),Theta)
	  Theta(1)=1-BD(j)/PD(j)
	  call VGpar(Pgupta,Theta,13,PG,4,50)
	  	  vgm=1-1/vgn
	  Gcap=vGG(vgalpha,vgm)
	  end if
	  call svpar(PG,IS,SavePar)


C PTF15 Rajkai and Varallyay, 1992
	  If(Clay(j)<0.or.Sand(j)<0.or.BD(j)<0.or.OC(j)<0) then
	  call nodata(vgthr,vgths,vgalpha,vgn,vgm,Gcap,IM,15,j)
											  else
	  Call Rajkai(Clay(j),Sand(j),BD(j),OC(j),Theta)
	  call VGpar(Prajkai,Theta,8,PG,4,50)
	  vgm=1-1/vgn
	  Gcap=vGG(vgalpha,vgm)
	  end if
	  call svpar(PG,IS,SavePar)

C PTF16 Rawls et al, 1983
	  If(Clay(j)<0.or.Silt(j)<0.or.Sand(j)<0.or.BD(j)<0.or.
     &  OC(j)<0) then
	  call nodata(vgthr,vgths,vgalpha,vgn,vgm,Gcap,IM,16,j)
											  else
	  Call Rawls83(Clay(j),Silt(j),Sand(j),BD(j),OC(j),Theta)
	  Theta(1)=1-BD(j)/PD(j)
	  call VGpar(Prawls03,Theta,10,PG,4,50)
	  vgm=1-1/vgn
	  Gcap=vGG(vgalpha,vgm)
	  end if
	  call svpar(PG,IS,SavePar)

	  write(13,'(100("|"F9.4))') (SavePar(i),i=1,89)
	  do i=1,16
	     k=5*(i-1)
	     if(i>8) k=6*(i-8)+35
	     do m=1,4
	        HyPar(i,j,m)=SavePar(k+m)
	        if(i<8.and.m==3) HyPar(i,j,m)=1/SavePar(k+m)
	     end do
        end do
	Enddo
	Write(11,'(A)') 'Brooks and Corey (1964) water retention model'
	Do i=1,16
      if(i==8) 	Write(11,'(/A)') 
     *         'van Genuchten (1980) water retention model'
	write(11,'(A30)') Title(i)
	Write(11,'(A)') ' ID  Depth    ThetaR    ThetaS    alpha       n'
	do j=1,nd
	If(IM(i,j)>0) then
	write(11,'(I4,F6.0,8F10.5)') ID(j),Depth(j),(HyPar(i,j,k),k=1,4)
	end if
	end do
	end do


C__________________________________________________________________________
C PTF17 Peterson et al., 1968
      write(12,'(/A)') 'Peterson et al., 1968'
	write(12,'(A,2F9.1)') 'Z/P,cm',P(5),P(9)
	Do j=1,nd
	  if(Clay(j)>0) then
			Call Peterson(Clay(j),t33,t1500)
			write(12,'(F5.1,2F9.3)') Depth(j),t33,t1500
	  end if
	Enddo

C PTF18 Bruand et al., 1994
      write(12,'(/A)') 'Bruand et al., 1994'
	write(12,'(A,2F10.1)') 'Z/P,cm',P(5),P(9)
	Do j=1,nd
	  if(Clay(j)>0) then
		  Call Bruand(Clay(j),t33,t1500)
		  write(12,'(F5.1,2F10.3)') Depth(j),t33,t1500
	end if
	Enddo

C PTF19 Canarache, 1993
      write(12,'(/A)') 'Canarache, 1993'
	write(12,'(A,2F9.1)') 'Z/P,cm',P(5),P(9)
	Do j=1,nd
	  if(Clay(j)>0.and.BD(j)>0) then
		  Call Canarache(Clay(j), BD(j), t33,t1500)
		  write(12,'(F5.1,2F9.3)') Depth(j),t33,t1500
	  end if
	Enddo

C PTF20 Hall et al., 1977
      write(12,'(/A)') 'Hall et al., 1977'
	write(12,'(A,2F9.1)') 'Z/P,cm',P(5),P(9)
	Do j=1,nd
	  if(Clay(j)>0.and.Silt(j)>0.and.Sand(j)>0.and.BD(j)>0) then
		  Call 	Hall(Clay(j),Silt(j),Sand(j),BD(j),t33,t1500)
		  write(12,'(F5.1,2F9.3)') Depth(j),t33,t1500
	  end if
	Enddo


      Close(11)
      Close(12)
	Close(13)
      Stop
	End



C=========================
C  PTF estimation ends
C=========================

 	Function BCG(h,Lambda)
	Real Lambda
	BCG=h*(2+3*Lambda)/(1+3*Lambda)
	Return
	End

      Subroutine BrooksCorey(Phi,BCLambda,hb,ThetaR,h,t)
	 if(h.LT.hb) then
	   t=Phi
	 else
	   t=ThetaR+(Phi-ThetaR)*(hb/h)**BCLambda
	 endif
	Return
	End

      Subroutine Bruand(Clay,t33,t1500)
      t33=    (0.043+0.4E-02*Clay)/(0.471+0.411E-02*Clay)
      t1500=  (0.008+0.367E-02*Clay)/(0.471+0.411E-02*Clay)
      Return
      End

      subroutine Campbell(Clay,Sand,BD,PD,ps,ts,b)
	 SILT=100.-CLAY-SAND
         sf=SILT/100.
         cf=CLAY/100.
c	   dg=exp(-0.025-3.63*sf-6.88*cf)
	   dg=exp(-0.80-3.17*sf-7.61*cf)
         sg=(exp(13.32*sf+47.7*cf-(alog(dg))**2))**0.5
         pes=0.05/sqrt(dg)
         b=-20*(-pes)+0.2*sg
         ps=pes*(BD/1.3)**(0.67*b)
         ts=1.-BD/PD
      return
      end

      Subroutine Canarache(Clay, BD, t33,t1500)
	t1500=BD*(0.2805*Clay+0.0009615*Clay*Clay)/100.
	t33=BD*(2.65+1.105*Clay - 0.01896*Clay*Clay 
     C          + 0.0001678*Clay*Clay*Clay+15.12*BD - 6.745*BD*BD 
     D          - 0.1975*Clay*BD)/100.
	Return
	End

      Subroutine GuLar(Clay,Silt,Sand,OC,BD,Theta)
c Clay, silt, sand, and OM in %
      Real a(5,12),Theta(14)
	 Data a/7.053,10.242,10.07,6.333,-321.2,
     2		5.678,9.228,9.135,6.103,-269.6,
     3		5.018,8.548,8.833,4.966,-242.3,
     4		3.89,7.066,8.408,2.817,-187.8,
     5		3.075,5.886,8.039,2.208,-143.4,
     6		2.181,4.557,7.557,2.191,-92.76,
     7		1.563,3.62,7.154,2.388,-57.59,
     8		0.932,2.643,6.636,2.717,-22.14,
     9		0.483,1.943,6.128,2.925,-2.04,
     A        0.214,1.538,5.908,2.855,15.3,
     B		0.076,1.334,5.802,2.653,21.45,
     C		-0.059,1.142,5.766,2.228,26.71/
	OM=OC*1.724
      Do i=1,12
      Theta(i+1)=(SAND*a(1,i)+Silt*a(2,i)+CLAY*a(3,i)+OM*a(4,i)
     *					+BD*a(5,i))/1000.
      Enddo
      Return
      End

	Subroutine Hall(Clay,Silt,Sand,BD,t33,t1500)
	t33=(20.81+0.45*Clay+0.13*Silt-5.95*BD)/100.
	t1500=(1.48+0.84*Clay-0.0055*Clay*Clay)/100.
	Return
	End


      Subroutine MAYR(Clay,Silt,Sand,BD,OC,a,b,Phi)
      a=exp(-4.9840297533 + 0.0509226283*sand + 0.1575152771*silt+
     A        0.1240901644*BD- 0.1640033143*OC - 0.0021767278*silt**2+
     B        0.0000143822 * silt**3+ 0.0008040715 * clay**2 + 
     C        0.0044067117 * OC**2)
      b=1.0/exp( -0.8466880654 - 0.0046806123*sand + 0.0092463819*silt -
     A         0.4542769707*BD - 0.0497915563*OC + 0.0003294687*sand**2-
     B          0.000001689056*sand**3 + 0.0011225373 * OC**2)
      Phi=0.2345971971+0.0046614221*Sand+0.0088163314*Silt
     A	+0.0064338641*Clay-0.3028160229*BD+0.0000179762*Sand**2
     B	-0.00003134631*Silt**2
	return
	end
	
	
	Subroutine nodata(ThetaR,Phi,hb,BCLambda,a,Gcap,IM,i,j)
	Integer IM(21,100)
	IM(i,j)=0
	a=-1
	ThetaR=-1
	Phi=-1
	hb=-1
	BCLambda=-1
	Gcap=-1
	return
	end

      Subroutine Oosterveld(Clay,Sand,BD,Depth,A)
	A=BD*(35.36+0.644*Clay-0.251*Sand+0.045*Depth)/100.
	return
	End


      Subroutine Peterson(Clay,t33,t1500)
	t33 = (11.83+0.96*Clay-0.008*Clay*Clay)/100.
	t1500 = (1.74 + 0.76*Clay - 0.005*Clay*Clay)/100.
	return
	End


      Subroutine Rajkai(Clay,Sand,BD,OC,Theta)
	Real b(6,8), Theta(14)
	Data b/89.75,  -31.39,      0,    0.030,       0,       0,
     A       85.05,  -27.17,      0,   -0.024,       0,       0,
     B       78.58,  -23.94,      0,   -0.025,       0,       0,
     C       69.78,  -21.74,      0,        0,       0,  0.0011,
     F       20.87,    0.29,  -0.83,    0.030,       0,  0.0051,
     G        2.19,    0.52,   3.93,    -0.07,       0,       0,
     H        1.39,    0.36,      0,        0,       0,  0.220 ,
     K        0.73,       0,   0.32,        0,  0.0018,       0/
	Silt=100-Clay-Sand
      Do i=1,8
	select case (i)
		case (1) 
					X1=BD
					X2=Silt
		case (2,3)
					X2=Sand
		case(4)
					X2=Clay+Silt
		case(5)
					X1=Clay+Silt
					X2=Sand/Silt
		case(6,7)
					X2=OC*1.724
		case(8)
					X1=Clay
	End select
	Theta(i)=(b(1,i)+b(2,i)*X1+b(3,i)*X2+b(4,i)*X1*X2+b(5,i)*X1*X1
     *		 +b(6,i)*X2*X2)/100.
	End do
      Return
      End


      Subroutine Rawls85(Clay,Sand,BD,PD,Phi,BCLambda,hb,ThetaR)

*  It reads Clay content(%), Sand content(%), Bulk density(g cm-3) and produces
*  porosity value Phi and also Brooks-Corey parameters Lambda, hb (cm), Theta
c
c                    Porosity
c
      Phi = (1. - BD/PD)

c                    Auxiliary values
c
         C2  = Clay**2
         S2  = Sand**2
         Phi2= Phi **2
c
c                    Lambda
c
      BCLambda = -0.7842831+0.0177544*Sand-1.062498*Phi-0.00005304*S2
     #          -0.00273493*C2+1.11134946*Phi2-0.03088295*Sand*Phi
     #          +0.00026587*S2*Phi2-0.00610522*C2*Phi2
     #          -0.00000235*S2*Clay+0.00798746*C2*Phi
     #          -0.00674491*Phi2*Clay
      BCLambda=exp(BCLambda)
c
c                    hb
c
      hb = 5.3396738+0.1845038*Clay-2.48394546*Phi-0.00213853*C2
     #    -0.04356349*Sand*Phi-0.61745089*Clay*Phi+0.00143598*S2*Phi2
     #    -0.00855375*C2*Phi2 -0.00001282*S2*Clay+0.00895359*C2*Phi
     #    -0.00072472*S2*Phi  +0.0000054*C2*Sand+0.50028060*Phi2*Clay
      hb=exp(hb)
c
c                   ThetaR
c
      ThetaR=-0.0182482+0.00087269*Sand+0.00513488*Clay+0.02939286*Phi
     #       -0.00015395*C2-0.0010827*Sand*Phi-0.00018233*C2*Phi2
     #       +0.00030703*C2*Phi-0.0023584*Phi2*Clay
c
	Return
	End

      Subroutine Rawls82(Clay,Silt,Sand,OC,Theta)
c Clay, silt, sand, and OM in %
      Real a(5,10), Theta(14)
C     The coefficients are taken from Saxton et al, SSSAJ 1986
	Data a/0.4118 ,  -0.0030  ,       0   , 0.0023 ,  0.0317,
     A       0.3121 ,  -0.0024  ,       0   , 0.0032 ,  0.0314,
     B       0.2576 ,  -0.0020  ,       0   , 0.0036 ,  0.0299,
     C       0.2065 ,  -0.0016  ,       0   , 0.0040 ,  0.0275,
     D       0.0349 ,        0  ,   0.0014  , 0.0055 ,  0.0251,
     E       0.0281 ,        0  ,   0.0011  , 0.0054 ,  0.0220,
     F       0.0238 ,        0  ,   0.0008  , 0.0052 ,  0.0190,
     G       0.0216 ,        0  ,   0.0006  , 0.0050 ,  0.0167,
     H       0.0205 ,        0  ,   0.0005  , 0.0049 ,  0.0154,
     K       0.0260 ,        0  ,       0   , 0.0050 ,  0.0158/
C	OM=OC*1.724
      Do i=1,10
      Theta(i+1)=a(1,i)+SAND*a(2,i)+Silt*a(3,i)
     $        +CLAY*a(4,i)+OC*a(5,i)
      Enddo
      Return
      End

      Subroutine Rawls83(Clay,Silt,Sand,BD,OC,Theta)
      Real a(5,9), Theta(14)
	Data a/0.4180 ,  -0.0021  , 0.0035 ,  0.0232, -0.0859,   
     2       0.3486 ,  -0.0018  , 0.0039 ,  0.0228, -0.0738,
     3       0.2819 ,  -0.0014  , 0.0042 ,  0.0216, -0.0612,
     4       0.2352 ,  -0.0012  , 0.0043 ,  0.0202, -0.0517,
     5       0.1837 ,  -0.0009  , 0.0044 ,  0.0181, -0.0407,
     6       0.1426 ,  -0.0007  , 0.0045 ,  0.0160, -0.0315,
     7       0.1155 ,  -0.0005  , 0.0045 ,  0.0143, -0.0253,
     8       0.1005 ,  -0.0004  , 0.0044 ,  0.0133, -0.0218,
     9       0.0854 ,  -0.0004  , 0.0044 ,  0.0122, -0.0182/
C	OM=OC*1.724
      Do i=1,9
      Theta(i+1)=a(1,i)+SAND*a(2,i)+Clay*a(3,i)+OC*a(4,i)+BD*a(5,i)
      Enddo
      Return
      End

C Saving water retention parameters to Excel file
	Subroutine svpar(Par,IS,SavePar)
	Real Par(6),SavePar(200)
        Do j=1,5
	       SavePar(IS)=Par(j)
	    IS=IS+1
	  End do
      if (IS>40) then
	    SavePar(IS)=Par(j)
	    IS=IS+1
	end if
	Return
	End



	Subroutine Saxton(Clay,Sand,AA,BB)
        a = -4.396
	  b = -0.0715
	  c = -4.88E-04
	  d = -4.285E-05
	  e = -3.14
	  f = -2.22E-03
	  g = -3.484E-05
	  AA = 100.0*exp(a+b*Clay+c*Sand*Sand+d*Sand*Sand*Clay) 
	  BB = e+f*Clay*Clay+g*Sand*Sand*Clay
	  Return
	End
c

      Subroutine Tomasella(Clay,Silt,OC,Theta)
	Real a(4,9),Theta(14)
	Data a/ 37.937,2.24,0.298,0.159,
	2		23.839,0, 0.53,0.255,
	3		18.495,0,0.552,0.262,
	4		12.333,0,0.576,  0.3,
	5		 9.806,0,0.543,0.321,
	6		 4.046,0,0.426,0.404,
	7		 3.198,0,0.369,0.351,
	8		 1.567,0,0.258,0.361,
	9		  0.91,0, 0.15,0.396/

      Do i=1,9
      Theta(i)=(a(1,i)+a(2,i)*OC+a(3,i)*Silt+a(4,i)*CLAY)/100.
      Enddo
	return
	End

      Function vangen(ts,tr,alp,an,am,h)
	vangen = (ts - tr)/(1.+ (alp*h)**an)**am+tr
	Return
	End

	Function vGG(alpha,m)
	Real m
	vGG=(0.046*m+2.07*m*m+19.5*m**3)/(1+4.7*m+16*m*m)/alpha
	Return
	End

      Subroutine Varallyay(Clay,Sand,BD,ts,an,pfx)
	C=Clay/100.
	S=Sand/100.
	ts=(-56.4*BD+20.5*C*C+123.7)/100.
	an = 0.336*BD-0.053
      pfx=1.51*BD+4.27*BD*C-0.417
	return
	End

	Subroutine Vereckeen(Clay,Sand,BD,OC,ts,tr,alp,an)
      ts = 0.81 -0.283*BD + 0.001*Clay
	tr = 0.015 + 0.005 *Clay +0.014*OC
      alp= exp(-2.486 + 0.025*sand - 0.351*clay)
	an = exp(0.053 - 0.009*sand - 0.013*clay +0.00015*sand*sand)
	Return
	End


      Subroutine Williams1(Clay,Sand,BD,Str,A,B)
C Function # 1 All sand is fine sand and the Structural index =1
      A=1.839+0.257*alog(CLAY)+0.3812*Str-0.0001*Sand*Sand
      B=-0.303+0.093*alog(BD)+0.0565*alog(CLAY)-0.00003*SAND*SAND
      return
      end
      
      Subroutine Williams2(Clay,Sand,OC,A,B)
C Function # 2 with all sand = fine sand
      OM=OC*1.724
      S=Sand
      A=2.57+0.238*Alog(CLAY)-0.000192*S*S-0.0926*Alog(OM)+0.0412*OM
      B=-0.403+0.0871*Alog(CLAY)-0.00077*S
      return
      end

      Subroutine Wosten1(Clay,Silt,Sand,itop,vgths,vgthr,vgalpha,
     $                   vgn,vgm)
	Real params(7,10)
	data params/0.025,0.403,0.0383,1.3774,0.2740, 1.2500,60.000,
     A            0.010,0.439,0.0314,1.1804,0.1528,-2.3421,12.061,
     B            0.010,0.430,0.0083,1.2539,0.2025,-0.5884, 2.272,
     C            0.010,0.520,0.0367,1.1012,0.0919,-1.9772,24.800,
     D            0.010,0.614,0.0265,1.1033,0.0936, 2.5000,15.000,
     E            0.025,0.366,0.0430,1.5206,0.3424, 1.2500,70.000,
     F            0.010,0.392,0.0249,1.1689,0.1445,-0.7437,10.755,
     G            0.010,0.412,0.0082,1.2179,0.1789, 0.5000, 4.000,
     I            0.010,0.481,0.0198,1.0861,0.0793,-3.7124, 8.500,
     J            0.010,0.538,0.0168,1.0730,0.0680, 0.0001, 8.235/
C==========
C FAO Class
C==========
      if(clay.GT.60.) then 
	  iclass=5
	elseif(clay.GT.35.) then 
	  iclass=4
	elseif(sand.LT.15.) then
	  iclass=3
	elseif(sand.GT.65.0.AND.clay.LT.18.0) then
	  iclass=1
	else
	  iclass=2
	endif
	j=iclass+itop*5
      vgths  =params(2,j)
	vgthr  =params(1,j)
	vgalpha=params(3,j)
	vgn    =params(4,j)
	vgm    =params(5,j)
	Return
	End

      Subroutine Wosten2(Clay,Silt,BD,OC,itop,vgths,vgthr,
     $                   vgalpha,vgn)
	OM=OC*1.724
        vgths= 0.7919 + 0.001691*clay - 0.29619*BD - 0.000001491*silt**2
     A +0.0000821*OM**2+0.02427/clay+0.01113/silt+0.01472*ALOG(silt)-
     B 0.0000733*OM*clay-0.000619*BD*clay-0.001183*BD*OM-
     C 0.0001664*itop*silt
        vgalpha=exp(-14.96+0.03135*clay+0.0351*silt+0.646*OM+15.29*BD- 
     A      0.192*itop - 4.671*BD**2 - 0.000781*clay**2 - 0.00687*OM**2+
     B      0.0449/OM + 0.0663*ALOG(silt) + 0.1482*ALOG(OM) - 
     C      0.04546*BD*silt - 0.4852*BD*OM +0.00673*itop*clay)
	 vgn=1.+exp(-25.23-0.02195*clay+0.0074*silt-0.1940*OM+45.5*BD
     A        - 7.24*BD**2 + 0.0003658*clay**2 + 0.002885*OM**2 
     B        -12.81/BD - 0.1524/silt - 0.01958/OM - 0.2876*ALOG(silt)
     C        - 0.0709*ALOG(OM) - 44.6*ALOG(BD) - 0.02264*BD*clay 
     D        + 0.0896*BD*OM + 0.00718*itop*clay)
        vgthr=0.01
	Return
	End


	  