%% Program calculate enzyme activity based 
% on the largest changes of grayscale values 
% over time in pixels of TLZ images
% Time (if irregular)is in input file Time.dat
% Activity units MUF units/mm2/min
% Linear regression fit to MUF dinamics with test for R2 
% activity set to zero in pixels with low or highly oscillating G-values
% activity multiplied by the corretion factor to account for the MUF losses due to diffusion to soil
% input > *.tif - a sequence of 8-bit tif files after background subtraction and median filtering (ImageJ)
%                 located in myFolder.
%       > Par.dat - file with calibration parameters (two columns). Use space as a delimeter.
%
% output < Activity.res - text image file with enzyme activities before corretion
%        < Fraction.res - text image file with the correction factors
%        < Time_der.res - text image file with time moments, when the activity was the highest.
%        < R2.res  - text image file with R2 of dMUF/dTime derivative calculation
%        < Res.txt - output file with list of sequense files and number of pixels



clear;
close all;
% Set the directory with MUF images 
myFolder = 'd:\TLZ\Examples\beta-glucosidase\MUF_filt\MUF_filt';

if exist(myFolder, 'dir') ~= 7
  Message = sprintf('Error: The following folder does not exist:\n%s', myFolder);
  uiwait(warndlg(Message));
  return;
end
filePattern = fullfile(myFolder, '*.tif');  % 8-bit image files in TIF format 
filePar = fullfile(myFolder, 'Par.dat');    % File with calibration parameters and time
tifFiles   = dir(filePattern);              % List of TIF-file names
fileRes = fullfile(myFolder,'Res.txt');     % output file name with list of sequense files
resfileAct = fullfile(myFolder,'Activity.res');% text image file with enzyme activities
resfileR2 = fullfile(myFolder,'R2.res');    % text image file with enzyme activities
resfileTder = fullfile(myFolder,'Time_der.res');    % text image file with times of peak activities
resfileFract = fullfile(myFolder,'Fraction.res');      % text image file with fractions of activities calculated from peak activities
fileID1 = fopen(fileRes,'w'); 
Par = dlmread(filePar,' ',0,1);             % read calibration paprameters from Par.dat file

%% Calibration parameters a1 and a2 [pmol/pix], b1 and b2 [pmol/pix/greyscale] 
b1 = Par(1,1);
a2 = Par(2,1);
b2 = Par(3,1);
Go = Par(4,1);

% Image processing parameters
nt = Par(5,1); % number of points for LR: [3-5]
R2min = Par(6,1); % R2 for acceptable LR: [0.7-0.9]
Res = Par(7,1); % image resolution, micrometers: [15-30]
SliceStep = 1;  % image increment for large number of images 
Time_step = Par(8,1);  % time interval between images in input directory
Gmin = Par(9,1);       % Min Greyscale in the pixel to run enzymatic activity
SlopeMin = Par(10,1);   % Min enzymatic activity based on the whole curveG
SigmaGmin = Par(11,1);   % Min standard deviation in G curve


%% Read image sequence
Tfiles = length(tifFiles);
for k = 1:Tfiles
  baseFileName = tifFiles(k).name;
  fullFileName = fullfile(myFolder, baseFileName);
  fprintf('Now reading %s\n', baseFileName);

  Im{k} = imread(fullFileName);
  
  sz=size(Im{1});
  fprintf(fileID1,'%s%8.0f%8.0f\n',baseFileName,sz(1),sz(2));
end

Pixel_area = Res*Res/1E+6;  % Pixel area in mm
Nfiles = Tfiles/SliceStep;  % Correct Time.dat if SliceStep>1 
% irregular time step, read time from Time.dat file.
if Time_step == 0     
    fileTime = fullfile(myFolder, 'Time.dat');    % File with time
    time = dlmread(fileTime,' ',0,0); 
    Tend = size(time,2);
    if Tfiles ~= Tend 
        Message = sprintf('Error: Total number of time records differes from number of images in: %s %6d %6d \n', myFolder, Tend, Tfiles);
        uiwait(warndlg(Message));
        return;
    end

else
    delT = SliceStep*Time_step; % Time increment for processing
    Tend = delT*(Nfiles-1);     % Total time length
    time = 0:delT:Tend;         % Time moments
end

% Merge all iamges in one 3D array
FileNum = 1:Tfiles;         % Slice numbers in the stack
Sequence = cat(3,Im{:});  % 3D array (sz(1),sz(2),nslices)

% set up array dimensions 
act = zeros(sz(1),sz(2));
actCor = zeros(sz(1),sz(2));
Corr = zeros(sz(1),sz(2));
Tder = zeros(sz(1),sz(2));
Fract = zeros(sz(1),sz(2));
MUF = zeros(1);
coef = zeros(1);
G = zeros(1);
Gshape = zeros(1);

%% Run calculations for selected pixel 
for j=1:sz(1)
  for k=1:sz(2)
     Gshape = double(squeeze(Sequence(j,k,:)));
     if max(Gshape) > Gmin

% Calculate MUF  [pmole/pixel]
      Grey = double(squeeze(Sequence(j,k,1:SliceStep:Tfiles)));

      for l=1:Nfiles
        if Grey(l) < Go % Go is the grey value at the breakpoint
            MUF(l) = b1*Grey(l); % Par(1)=a1 should be zero when backround subtracted 
        else
            MUF(l) = a2 + b2*Grey(l);  % Oversaturated part of the calibration line
        end
        if MUF(l) < 0
            MUF(l)= 0.0;
        end  
      end  
      
 % G(time)-slope for the whole curve G(t)
    t=time.';
    Tf = [ones(length(t),1) t];
    coef = Tf\Grey;
    Gcal=coef(1)+coef(2)*t;
    SigmaG =std(Grey); % standard deviation of G-values
        if SigmaG > SigmaGmin && coef(2) > SlopeMin 
        % well defined increase in G-values activity > 0 
            slope = 0;
            R2max = 0.0;
            R2 = 0;
            Tder(j,k) = 0;
        % Derivative dMUF/dTime calculations for nt time points with sliding window.
            for i=1:(Nfiles-nt+1)
                i1=i;
                i2=i1+nt-1;
                t=time(i1:i2).';
                G=MUF(i1:i2).';
                Tf = [ones(length(t),1) t];
                coef = Tf\G;
                Gcal=coef(1)+coef(2)*t;
                Sres= sum((G - Gcal).^2);
                Stot=sum((G - mean(G)).^2);
                if Stot > 1.E-10
                    R2 = 1 - Sres/Stot;
                    if (R2 > R2min) && (coef(2) > slope) 
                       slope = max(slope,coef(2)); % maximum slope
                       R2max = max(R2max,R2);
                       Tder(j,k) = (time(i1)+time(i2))/2; % Time moment when the activity was the highes
                    end 
                end
            end
        else
        % Oscillations in G-values without increse over time = zero activity    
            slope = 0; 
            R2max = 0;
            Tder(j,k) = 0;
        end

     else
         % G-values are too small  = zero activity
            slope = 0; 
            R2max = 0;
            Tder(j,k) = 0;
        end

        act(j,k) = slope/Pixel_area; 
        Corr(j,k) = R2max;
        if Tder(j,k) > 0 
            % The correction factor for MUF losses
            Fract(j,k) = 100/(2.947+25.95/(0.840+Tder(j,k))); 
        end
    end
     run = 100*(sz(2)*(j-1)+k)/(sz(1)*sz(2));
     fprintf('run = %6.1f\n ',run); % Percentage of processed pixels
end

%% Save Results in ImageJ as Text Image
dlmwrite(resfileAct,act);
dlmwrite(resfileR2,Corr);
dlmwrite(resfileTder,Tder);
dlmwrite(resfileFract,Fract);
fclose(fileID1);

%% Statistics of enzyme activity
X = Tder(:);
Yact = act(:);
actCor = act.*Fract;
Ycor = actCor(:);
t = time;

fileStat = fullfile(myFolder, 'Stat.dat'); 
fileID2 = fopen(fileStat,'w');
nY = zeros(1);
Ymean = zeros(1);
Ystdev =zeros(1);
Ybound = zeros(1);
tplot = zeros(1);
Ymcor= zeros(1);
for i=1:Nfiles-1
  idxX = find((X>t(i) & X<=t(i+1)));
  Yres = Yact(idxX);
  nY(i) = numel(Yres);
  if nY(i) > 0 
      Ymean(i) = mean(Yres);
      Ystdev(i) = std(Yres);
      Ymcor(i) = mean(Ycor(idxX));
  else
      Ymean(i) = 0;
      Ystdev(i) = 0;
      Ymcor(i) = 0;
  end    
  tplot(i) = (t(i+1)+t(i))/2;
  fprintf(fileID2,'%6.2f %6d %8.4f %8.4f\n', tplot(i),nY(i),Ymean(i),Ystdev(i));
end
fclose(fileID2);
%% Plot Number of active pixels and peak activities vs. time 
 f1 = figure();
 f2 = figure(); 
 f3 = figure();
 figure(f1);

% title('Number of active pixels');
scatter(tplot,nY);
xlabel('time, min') 
ylabel('Number of active pixels')

fileFig1 = fullfile(myFolder,'Fig-1.jpg');
fileFig1M = fullfile(myFolder,'Fig-1m.fig');
saveas(f1,fileFig1);
saveas(f1,fileFig1M);

 figure(f2);
 scatter(tplot,Ymean);
% title('Enzyme activity');
 xlabel('time, min') 
 ylabel('Enzyme activity, pmol/mm2/min') 
 hold on
 Ybound = Ymean - Ystdev;
 plot(tplot,Ybound);
 hold on
 Ybound = Ymean + Ystdev;
  plot(tplot,Ybound);
  hold off
  fileFig2 = fullfile(myFolder,'Fig-2.jpg');
  saveas(f2,fileFig2);

  figure(f3);
  scatter(tplot,Ymcor);
% title('Enzyme activity');
 xlabel('time, min') 
ylabel('Corrected enzyme activity, pmol/mm2/min') 
fileFig3 = fullfile(myFolder,'Fig-3.jpg');
saveas(f3,fileFig3);

  
%%



